<?php

namespace App\Services\Master;

use App\Http\Resources\Master\MasterWorkerResource;
use App\Models\PtwMstWorker;
use App\Models\PtwMstWorkerDsg;
use App\Models\PtwMstContractor;
use App\Models\PtwTypeMapWorkers;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;

class MasterWorkerServices extends MasterWorkerResource
{
    public $vvsPtwMstWorkerModel;
    public $page;
    public $pageSize;
    public function __construct() {
        $this->vvsPtwMstWorkerModel = new PtwMstWorker();
    }

    /**
     * Column name in DB Table and Front end request variables are mapped
     * key is the column name of DB table and value is the front end request variable
     */
    protected function columnMappingsFrontEndVsBackEnd()
    {
        return [
            'WORKER_NAME' => 'WORKER_NAME',
            'WORKER_DESG_ID' => 'WORKER_DESG_ID',
            'WORKER_CONTRACTOR_ID' => 'WORKER_CONTRACTOR_ID',
        ];
    }
    
    public function getMasterWorkerList($requestData) {
        $columnMapping = $this->columnMappingsFrontEndVsBackEnd();
        $site_id = $requestData->SITE_ID;
        $query = PtwMstWorker::select(
            'PTW_MST_WORKERS.WORKER_ID',
            'PTW_MST_WORKERS.SITE_ID',
            'PTW_MST_WORKERS.WORKER_NAME',
            'PTW_MST_WORKERS.WORKER_AUTO_GEN_ID',
            'PTW_MST_WORKER_DSG.WORKER_DESG_NAME',
            'PTW_MST_CONTRACTOR.CONTRACTOR_NAME',
            'PTW_MST_WORKERS.IS_ACTIVE',
            'PTW_MST_WORKERS.IS_DELETED',
        )
        ->with(['typeMappings.type'])
        ->leftJoin('PTW_MST_WORKER_DSG', function ($join) {
            $join->on('PTW_MST_WORKERS.WORKER_DESG_ID', '=', 'PTW_MST_WORKER_DSG.WORKER_DESG_ID');
        })
        ->leftJoin('PTW_MST_CONTRACTOR', function ($join) {
            $join->on('PTW_MST_WORKERS.WORKER_CONTRACTOR_ID', '=', 'PTW_MST_CONTRACTOR.CONTRACTOR_ID');
        })
        
        ->where('PTW_MST_WORKERS.IS_DELETED', 0)
        ->where('PTW_MST_WORKERS.SITE_ID', $site_id)
        ->orderBy("WORKER_ID", "DESC");

        // Iterate over the requested variables
        foreach ($requestData as $key => $value) {
            // Check if the value is not empty and the mapping exists
            if (!empty($value) && isset($columnMapping[$key])) {
                // Special handling for WORKER_NAME to make it case-insensitive
            if ($key === 'WORKER_NAME') {
                $query->where($columnMapping[$key], 'ILIKE', "%$value%");
            } else {
                // Add a where clause for the mapped column name
                $query->where($columnMapping[$key], $value);
            }
            }
        }

        $totalCount = $query->count('PTW_MST_WORKERS.WORKER_ID');

         // Pagination Logic
        $this->page = $requestData->PAGE_NUM ?? 1; // Default to page 1 if not provided
        $this->pageSize = $requestData->PAGE_SIZE ?? config("constants.PAGINATION_PER_PAGE");

        $paginatedWorkerData = $query->paginate($this->pageSize, ['*'], 'page', $this->page);

        return [
            'data' => $paginatedWorkerData,
            'total' => $totalCount // Total count of distinct worker IDs
        ];
    }

    public function workerSave($requestData, $vvsUserRole)
    {
        $workerId = "";
        $processedWorkerData = [];
        if(isset($requestData -> WORKER_ID) && $requestData -> WORKER_ID){
            $workerdata = PtwMstWorker::where("WORKER_ID",$requestData -> WORKER_ID)->where("IS_DELETED",0)->first();
            $workerId = $workerdata->WORKER_ID;
        }
        
        DB::beginTransaction();
        try {
        if($requestData -> ACTION == "DELETE"){
            $typeCount = PtwTypeMapWorkerse::where("IS_DELETED", 0)->where("IS_ACTIVE", 1)
                ->whereIn("WORKER_ID", $requestData -> WORKER_ID)->count();
            if($typeCount > 0){
                $vvsCntMsg = "Worker cannot be deleted due to assign for Type";
                return $vvsCntMsg;
            }
        }
        if($requestData -> ACTION == "SAVE"){
            $uploadPath = './' . config('constants.WORKER_FILE_UPLOAD_PATH');
            $absoluteBasePath = public_path(config('constants.WORKER_FILE_UPLOAD_PATH'));
            if (!file_exists($uploadPath)) {
                mkdir($uploadPath, 0755, true);
            }
            $requestData->WORKER_NAME =  $requestData->WORKER_NAME ??  $workerdata->WORKER_NAME;
            $requestData->WORKER_DESG_ID =  $requestData->WORKER_DESG_ID ??  $workerdata->WORKER_DESG_ID;
            $requestData->WORKER_CONTRACTOR_ID =  $requestData->WORKER_CONTRACTOR_ID ??  $workerdata->WORKER_CONTRACTOR_ID;
            $processedWorkerData = [
                "SITE_ID" => (int)$requestData->SITE_ID,
                "WORKER_NAME" => $requestData->WORKER_NAME,
                "WORKER_DESG_ID" => (int)$requestData->WORKER_DESG_ID,
                "WORKER_CONTRACTOR_ID" => (int)$requestData->WORKER_CONTRACTOR_ID
            ];
            $fileList = (isset($requestData->WORKER_FILES)) ? $requestData->WORKER_FILES : [];

            if (count($fileList)) {
                $base64string = $fileList[0]->FILE;
                if( $base64string ){
                    list($type, $imageBase64) = explode(";base64,", $base64string);
                    list(, $extension) = explode('/', $type);

                    $newFileName = uniqid() . '.' . $extension;
                    $filePath = $absoluteBasePath . $newFileName;

                    if(File::put($filePath, base64_decode($imageBase64))){
                        $processedWorkerData["FILE_NAME"] = $newFileName;
                        $processedWorkerData["FILE_CAPTION"] = $fileList[0]->FILE_CAPTION;
                    }
                }
            }
        }

        if($requestData -> ACTION == "APPROVE"){
            $processedWorkerData["IS_ACTIVE"] = 1;
        }
        if($requestData -> ACTION == "DELETE"){
            $processedWorkerData["IS_DELETED"] = 1;
        }
        
        if( isset($requestData -> WORKER_ID) &&  $requestData -> WORKER_ID > 0){
            $processedWorkerData["MODIFIED_BY"] = $requestData->CREATED_BY;
            $processedWorkerData["MODIFIED_ON"] = currentDateTime();
        }else{
            $processedWorkerData["CREATED_BY"] = $requestData->CREATED_BY;
            $processedWorkerData["CREATED_ON"] = currentDateTime();
        }
        
        if( $workerId > 0 ){
            //update PtwMstWorker table
            PtwMstWorker::where("WORKER_ID",$workerId)->update($processedWorkerData);
        }else{
            //insert into PtwMstWorker table
            $processedWorkerData["WORKER_AUTO_GEN_ID"] = $this -> autoWorkerGenIdGeneration($requestData);
            if ($vvsUserRole == 10) {
                $processedWorkerData["IS_ACTIVE"] = 1;
            } else {
                $processedWorkerData["IS_ACTIVE"] = 0;
            }
            // prd($processedWorkerData);
            $workerInsert = PtwMstWorker::create($processedWorkerData);
            $workerId = $workerInsert->WORKER_ID;
        }

        if($requestData -> ACTION == "SAVE"){
            //insert into PtwTypeMapWorkers
            $ptwTypeMapWorkersData = $this->transformPtwTypeMapWorkersTableData($requestData, $workerId);
            if(count($ptwTypeMapWorkersData)){
                //soft delete old data if exists
                PtwTypeMapWorkers::where("WORKER_ID", $workerId)->where("IS_DELETED", 0)
                                ->update(["IS_DELETED" => 1, "IS_ACTIVE"=>0, "MODIFIED_BY"=>$requestData->CREATED_BY,
                                            "MODIFIED_ON"=>currentDateTime()]);
                //insert new data
                PtwTypeMapWorkers::insert($ptwTypeMapWorkersData);
            }
        }

        $returnmsg = 'Processed successfully';
        $workerAutogenId = PtwMstWorker::where("WORKER_ID",$workerId)->pluck("WORKER_AUTO_GEN_ID")->first();
        if($requestData -> ACTION == "SAVE"){            
            if(isset($requestData -> WORKER_ID) && $requestData -> WORKER_ID){
                $returnmsg = 'Worker (ID '.$workerAutogenId.') data has been updated successfully.';
            }else{                
                $returnmsg = 'Worker (ID '.$workerAutogenId.') data has been created successfully.'; 
            }
        }
        if($requestData -> ACTION == "APPROVE"){
            $returnmsg = 'Worker (ID '.$workerAutogenId.') data has been approved successfully.';
        }
        if($requestData -> ACTION == "DELETE"){
            $returnmsg ='Worker (ID '.$workerAutogenId.') data has been deleted successfully.'; 
        }

        DB::commit();

         // If the transaction is committed successfully, you can return a response or perform other actions
         //return PtwMstWorker::where("WORKER_ID",$workerId)->pluck("WORKER_AUTO_GEN_ID")->first();
         return $returnmsg;
        
        } catch (\Exception $e) {
        // If an exception occurs during the transaction, rollback the changes
        DB::rollBack();

        // You can handle the exception here, log it, or return an error response
        return "";
        }
    }
    
    protected function autoWorkerGenIdGeneration($requestData)
    {
        $siteId = $requestData->SITE_ID;
        $contractorId = $requestData->WORKER_CONTRACTOR_ID;

        $allWorkerData = PtwMstWorker::select("WORKER_AUTO_GEN_ID")
                ->where("WORKER_AUTO_GEN_ID", "LIKE", "WORKER_IND_%")->orderBy("WORKER_ID", "DESC")->get();
        
        $prefix = config("constants.WORKER_REFERENCE_PREFIX");
        $previousNumber = 0;
        if (count($allWorkerData->toArray())) {
            $separation = explode('_', $allWorkerData[0]->WORKER_AUTO_GEN_ID);
            $previousNumber = (int)end($separation);
        }
        // Increment the last digit based on the value of $previousNumber
        $lastDigit = $previousNumber + 1;

        // Concatenate all parts to form the output
        return $prefix . $lastDigit;
    }

    protected function transformPtwTypeMapWorkersTableData($requestData, $workerId){
        $processedData = [];
        if(isset($requestData->WORKER_COMPETENCY)){
            if(count($requestData->WORKER_COMPETENCY)){
                foreach ($requestData->WORKER_COMPETENCY as $value) {
                    $processedData[] = [
                        "WORKER_ID" => $workerId,
                        "TYPE_ID" => $value,
                        "CREATED_BY" => $requestData->CREATED_BY,
                        "CREATED_ON" => currentDateTime()
                    ];
                }
            }
        }
        return $processedData;
    }

    public function getWorkerDetails($requestData)
    {
        return PtwMstWorker::where("WORKER_ID", $requestData->WORKER_ID)->first();
    }
}