<?php

namespace App\Services\Dashboard;

use App\Http\Resources\Dashboard\DashboardHiraResource;
use App\Models\MstSite;
use App\Models\User;
use App\Models\UserMapRole;
use App\Models\MstLocation;
use App\Models\Hira;
use DateTime;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;

class DashboardHiraServices extends DashboardHiraResource
{
    public $vvsHiraModel;
    
    public function __construct()
    {
        $this->vvsHiraModel = new Hira();
    }

    /**
     * Column name in DB Table and Front end request variables are mapped
     * key is the column name of DB table and value is the front end request variable
     */
    protected function columnMappingsFrontEndVsBackEnd()
    {
        return [
            'SITE_ID' => 'SITE_ID',
            'HIRA_START_DATE'=>'HIRA_START_DATE',
            'HIRA_END_DATE'=>'HIRA_END_DATE'
        ];
    }

    public function getDashboardHiraTotalCount($requestData)
    {
        $columnMapping = $this->columnMappingsFrontEndVsBackEnd();

        $query = Hira::where("IS_ACTIVE", 1)->where("IS_DELETED", 0)->whereNotIn("STAGE_ID", [1])
                    ->whereNotNull('STAGE_ID');
        
        // Check if SITE_ID is set and not equal to 0
        if (isset($requestData->SITE_ID) && $requestData->SITE_ID != 0) {
            $query->where('SITE_ID', $requestData->SITE_ID);
        }
        // Iterate over the requested variables
        foreach ($requestData as $key => $value) {
            // Check if the value is not empty and the mapping exists
            if (!empty($value) && isset($columnMapping[$key])) {
                // Add a where clause for the mapped column name
                if ($columnMapping[$key] == "HIRA_START_DATE") {
                    $query->whereDate($columnMapping[$key], '>=', $value);
                } elseif ($columnMapping[$key] == "HIRA_END_DATE") {
                    $query->whereDate($columnMapping[$key], '<=', $value);
                } elseif ($columnMapping[$key] != "SITE_ID") {
                    $query->where($columnMapping[$key], $value);
                }
            }
        }

        return $query->count();
    }

    public function getDashboardHiraByStageCount($requestData)
    {
        $columnMapping = $this->columnMappingsFrontEndVsBackEnd();

        // Create the initial subquery
        $query2 = DB::table('HIRA')->select('HIRA.STAGE_ID', DB::raw('COUNT("HIRA"."STAGE_ID") AS HIRA_COUNT'))
            ->where('HIRA.IS_ACTIVE', 1)
            ->where('HIRA.IS_DELETED', 0)
            ->whereNotNull('HIRA.STAGE_ID')
            ->whereNotIn("HIRA.STAGE_ID", [1]);

        // Check if SITE_ID is set and not equal to 0
        if (isset($requestData->SITE_ID) && $requestData->SITE_ID != 0) {
            $query2->where('HIRA.SITE_ID', $requestData->SITE_ID);
        }

        // Iterate over the requested variables for additional conditions
        foreach ($requestData as $key => $value) {
            if (!empty($value) && isset($columnMapping[$key])) {
                if ($columnMapping[$key] == "HIRA_START_DATE") {
                    $query2->whereDate($columnMapping[$key], '>=', $value);
                } elseif ($columnMapping[$key] == "HIRA_END_DATE") {
                    $query2->whereDate($columnMapping[$key], '<=', $value);
                }
            }
        }

        $query2->groupBy('HIRA.STAGE_ID');

        // Convert the subquery to a raw SQL string with bindings
        $sql2Query = getQueryWithBindings($query2->toSql(), $query2->getBindings());

        // Create the main query with the left join
        $query = DB::table('HIRA_MST_STAGE')->select(
            'HIRA_MST_STAGE.STAGE_ID',
            'HIRA_MST_STAGE.STAGE_NAME',
            'PTS.hira_count AS HIRA_COUNT'
        )
        ->leftJoin(DB::raw('('.$sql2Query.') AS "PTS"'), function ($join) {
            $join->on('PTS.STAGE_ID', '=', 'HIRA_MST_STAGE.STAGE_ID');
        })
        ->where('HIRA_MST_STAGE.IS_ACTIVE', 1)
        ->where('HIRA_MST_STAGE.IS_DELETED', 0)
        ->whereNotNull('HIRA_MST_STAGE.STAGE_ID')
        ->orderBy('HIRA_MST_STAGE.STAGE_ID');

        return $query->get();
    }

    public function getDashboardActivityByJobCount($requestData)
    {
        $columnMapping = $this->columnMappingsFrontEndVsBackEnd();

        // Create the initial subquery
        $query2 = DB::table('HIRA_ACTIVITY')
            ->select('HIRA_ACTIVITY.JOB_ID', DB::raw('COUNT("HIRA_ACTIVITY"."JOB_ID") AS HIRA_ACTIVITY_COUNT'))
            ->join('HIRA', 'HIRA.ID', '=', 'HIRA_ACTIVITY.HIRA_ID')
            ->where('HIRA_ACTIVITY.IS_ACTIVE', 1)
            ->where('HIRA_ACTIVITY.IS_DELETED', 0)
            ->whereNotNull('HIRA_ACTIVITY.JOB_ID')
            ->whereNotIn('HIRA.STAGE_ID', [1]);

        // Check if SITE_ID is set and not equal to 0
        if (isset($requestData->SITE_ID) && $requestData->SITE_ID != 0) {
            $query2->where('HIRA.SITE_ID', $requestData->SITE_ID);
        }

        // Iterate over the requested variables for additional conditions
        foreach ($requestData as $key => $value) {
            if (!empty($value) && isset($columnMapping[$key])) {
                if ($columnMapping[$key] == 'HIRA_START_DATE') {
                    $query2->whereDate('HIRA.HIRA_START_DATE', '>=', $value);
                } elseif ($columnMapping[$key] == 'HIRA_END_DATE') {
                    $query2->whereDate('HIRA.HIRA_END_DATE', '<=', $value);
                }
            }
        }

        $query2->groupBy('HIRA_ACTIVITY.JOB_ID');

        // Convert the subquery to a raw SQL string with bindings
        $sql2Query = getQueryWithBindings($query2->toSql(), $query2->getBindings());

        // Create the main query with the left join
        $query = DB::table('HIRA_MST_JOB')->select(
            'HIRA_MST_JOB.JOB_ID',
            'HIRA_MST_JOB.JOB_NAME',
            'PTS.hira_activity_count AS HIRA_ACTIVITY_COUNT'
        )
        ->leftJoin(DB::raw('('.$sql2Query.') AS "PTS"'), function ($join) {
            $join->on('PTS.JOB_ID', '=', 'HIRA_MST_JOB.JOB_ID');
        })
        ->where('HIRA_MST_JOB.IS_ACTIVE', 1)
        ->where('HIRA_MST_JOB.IS_DELETED', 0)
        ->whereNotNull('HIRA_MST_JOB.JOB_ID')
        ->orderBy('HIRA_MST_JOB.JOB_ID');

        return $query->get();
    }

    public function getDashboardActivityByRiskRatingCount($requestData)
    {
        $columnMapping = $this->columnMappingsFrontEndVsBackEnd();

        // Create the initial query
        $query = DB::table('HIRA_ACTIVITY')
            ->select(
                'HIRA_ACTIVITY.ORD_BY_SEVERITY',
                DB::raw('COUNT("HIRA_ACTIVITY"."ORD_BY_SEVERITY") AS "HIRA_ACTIVITY_COUNT"'))
            ->join('HIRA', 'HIRA.ID', '=', 'HIRA_ACTIVITY.HIRA_ID')
            ->where('HIRA_ACTIVITY.IS_ACTIVE', 1)
            ->where('HIRA_ACTIVITY.IS_DELETED', 0)
            ->whereNotNull('HIRA_ACTIVITY.ORD_BY_SEVERITY')
            ->whereNotIn('HIRA.STAGE_ID', [1])
            ->groupBy('HIRA_ACTIVITY.ORD_BY_SEVERITY')
            ->orderBy('HIRA_ACTIVITY.ORD_BY_SEVERITY');

        // Check if SITE_ID is set and not equal to 0
        if (isset($requestData->SITE_ID) && $requestData->SITE_ID != 0) {
            $query->where('HIRA.SITE_ID', $requestData->SITE_ID);
        }

        // Iterate over the requested variables for additional conditions
        foreach ($requestData as $key => $value) {
            if (!empty($value) && isset($columnMapping[$key])) {
                if ($columnMapping[$key] == 'HIRA_START_DATE') {
                    $query->whereDate('HIRA.HIRA_START_DATE', '>=', $value);
                } elseif ($columnMapping[$key] == 'HIRA_END_DATE') {
                    $query->whereDate('HIRA.HIRA_END_DATE', '<=', $value);
                }
            }
        }

        return $query->get();
    }

    public function getDashboardActivityByRiskMatrix($requestData)
    {
        $columnMapping = $this->columnMappingsFrontEndVsBackEnd();

        // Create the initial query
        $query = DB::table('HIRA_ACTIVITY')
            ->join('HIRA', 'HIRA.ID', '=', 'HIRA_ACTIVITY.HIRA_ID')
            ->join('HIRA_MST_SEVERITY', 'HIRA_MST_SEVERITY.SEVERITY_ID', '=', 'HIRA_ACTIVITY.SEVERITY_ID')
            ->join('HIRA_MST_LIKELIHOOD', 'HIRA_MST_LIKELIHOOD.LIKELIHOOD_ID', '=', 'HIRA_ACTIVITY.LIKELIHOOD_ID')
            ->select(
                'HIRA_ACTIVITY.SEVERITY_ID',
                'HIRA_ACTIVITY.LIKELIHOOD_ID',
                'HIRA_MST_SEVERITY.SEVERITY_NAME',
                'HIRA_MST_LIKELIHOOD.LIKELIHOOD_NAME',
                DB::raw('COUNT(*) AS "HIRA_ACTIVITY_COUNT"')
            )
            ->where('HIRA_ACTIVITY.IS_ACTIVE', 1)
            ->where('HIRA_ACTIVITY.IS_DELETED', 0)
            ->whereNotIn('HIRA.STAGE_ID', [1]);

        // Check if SITE_ID is set and not equal to 0
        if (isset($requestData->SITE_ID) && $requestData->SITE_ID != 0) {
            $query->where('HIRA.SITE_ID', $requestData->SITE_ID);
        }

        // Iterate over the requested variables for additional conditions
        foreach ($requestData as $key => $value) {
            if (!empty($value) && isset($columnMapping[$key])) {
                if ($columnMapping[$key] == 'HIRA_START_DATE') {
                    $query->whereDate('HIRA.HIRA_START_DATE', '>=', $value);
                } elseif ($columnMapping[$key] == 'HIRA_END_DATE') {
                    $query->whereDate('HIRA.HIRA_END_DATE', '<=', $value);
                }
            }
        }

        $query->groupBy(
            'HIRA_ACTIVITY.SEVERITY_ID',
            'HIRA_ACTIVITY.LIKELIHOOD_ID',
            'HIRA_MST_SEVERITY.SEVERITY_NAME',
            'HIRA_MST_LIKELIHOOD.LIKELIHOOD_NAME'
        );

        return $query->get();
    }
}